/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.ComponentModel;
using System.ComponentModel.Design.Serialization;// for type converter
using System.Globalization; // for type converter
using System.Reflection;// for type converter
using System.Drawing.Design; // for type editor

using Borland.Eco.UmlRt;
using Borland.Eco.ObjectRepresentation;
using Borland.Eco.Services;
using Borland.Eco.Globalization;

namespace Borland.Eco.Handles
{
	/// <summary>
	/// <para>Type converter for AbstractColumn. Mainly for the internal use of the WinForm designer.</para>
	/// </summary>
	public abstract class AbstractColumnConverter: ExpandableObjectConverter
	{
		public override bool CanConvertTo(ITypeDescriptorContext context, Type destinationType)
		{
			if (destinationType == typeof(InstanceDescriptor))
				return true;
			return base.CanConvertTo(context, destinationType);
		}

		protected abstract Type GetTargetType();

		public override object ConvertTo(ITypeDescriptorContext context, CultureInfo culture, object value, Type destinationType)
		{
			if (destinationType == typeof(InstanceDescriptor))
			{
				if (value != null)
				{
					ConstructorInfo ctor = GetTargetType().GetConstructor(new Type[0] {});
					return new InstanceDescriptor(ctor, new object[0] {}, false);
				}
			}
			return base.ConvertTo(context, culture, value, destinationType);
		}
	}

	/// <summary>
	/// <para>Type converter for Column. Mainly for the internal use of the WinForm designer.</para>
	/// </summary>
	public sealed class ColumnConverter: AbstractColumnConverter
	{
		protected override Type GetTargetType()
		{
			return typeof(Column);
		}
	}

	/// <summary>
	/// <para>Type converter for OclColumn. Mainly for the internal use of the WinForm designer.</para>
	/// </summary>
	public sealed class OclColumnConverter: AbstractColumnConverter
	{
		protected override Type GetTargetType()
		{
			return typeof(OclColumn);
		}
	}

	/// <summary>
	/// <para>Type converter for EventDerivedColumn. Mainly for the internal use of the WinForm designer.</para>
	/// </summary>
	public sealed class EventDerivedColumnConverter: AbstractColumnConverter
	{
		protected override Type GetTargetType()
		{
			return typeof(EventDerivedColumn);
		}
	}

	/// <summary>
	/// <para>Abstract superclass for "Columns", i.e. additional properties on the
	/// Objects in the bindinglist of an ElementHandle</para>
	/// </summary>
	/// <remark>
	/// The strange inheritance hierarchy is in order to keep compatibility with the old
	/// definition of 'Column'.
	/// </remark>
	[Serializable()]
	public abstract class AbstractColumn
	{
		private ColumnCollection m_Owner;
		private string m_Name = string.Empty;
		private bool m_Nested;
		private string m_NestingName = string.Empty;
		private bool m_IsReadOnly;

		protected AbstractColumn() {}

		internal ColumnCollection Owner
		{
			set
			{
				if ((m_Owner != null) && (m_Owner != value))
					throw new NotSupportedException();
				m_Owner = value;
			}
			get { return m_Owner; }
		}

		protected void Changed()
		{
			if (m_Owner != null)
				m_Owner.Changed();
		}

		private void CheckUniqueName(string name)
		{
			if (Owner == null) return;
			if (!Owner.NameIsUnique(name))
				throw new InvalidOperationException(HandlesStringRes.sNameNotUnique(name));
		}

		/// <summary>
		/// <para>This is the name given to the property added to the binding list.</para>
		/// <para>This name is used when binding to a data aware control.</para>
		/// </summary>
		///<exception cref="InvalidOperationException">Thrown if the name of the column is not unique in the owner's column collection.</exception>
		[LocalizableCategory(typeof(HandlesStringRes), "sCategoryBinding")]
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyColumnName")]
		[DefaultValue("")]
		public String Name
		{
			get { return m_Name; }
			set
			{
				if (m_Name != value)
				{
					NameValidator.AssertNameIsValid(value);
					CheckUniqueName(value);
					m_Name = value;
					Changed();
				}
			}
		}

		/// <summary>
		/// <para>Set this value to true to indicate that the property shall be read only.</para>
		/// <para>If this value is false for an OclColumn, the readablility will be decided by the Ocl expression</para>
		/// <para>If this value is false for an EventDerviedColumn, setting is handled by ElementHandle.ReverseDeriveValue</para>
		/// </summary>
		[LocalizableCategory(typeof(HandlesStringRes), "sCategoryBinding")]
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyIsReadOnly")]
		[DefaultValue(false)]
		public bool IsReadOnly
		{
			get { return m_IsReadOnly; }
			set
			{
				if (m_IsReadOnly != value)
				{
					m_IsReadOnly = value;
					Changed();
				}
			}
		}

		/// <summary>
		/// If this name is set, a corresponding name must exist in the
		/// NestingsCollection of the owning handle.
		/// </summary>
		[LocalizableCategory(typeof(HandlesStringRes), "sCategoryNesting")]
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyNesting")]
		[DefaultValue("")]
		public String NestingName
		{
			get { return m_NestingName; }
			set
			{
				if (m_NestingName != value)
				{
					m_NestingName = value;
					Changed();
				}
			}
		}

		/// <summary>
		/// <para> OCL expression used to compute the value of the column unless <see cref="EventDerivedValue"/> is true. </para>
		/// </summary>
		[Browsable(false)]
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public abstract string Expression { get; set; }

		[Browsable(false)]
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public abstract bool EventDerivedValue { get; set; }

		/// <summary>
		/// Set this property to true to make the property in the binding list a list rather than a single value.
		/// </summary>
		[LocalizableCategory(typeof(HandlesStringRes), "sCategoryNesting")]
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyNested")]
		[DefaultValue(false)]
		public bool Nested
		{
			get { return m_Nested; }
			set
			{
				if (m_Nested != value)
				{
					m_Nested = value;
					Changed();
				}
			}
		}
	}

	/// <summary>
	/// <para> Class describing an additional property to be added to the items in the rendered list defined via OCL</para>
	/// </summary>

	[Serializable()]
	[TypeConverter(typeof(OclColumnConverter))]
	public class OclColumn: AbstractColumn, IHasEditableExpression
	{
		private string m_Expression = string.Empty;

		#region Implementation of IHasEditableExpression
		///<exception cref="InvalidOperationException">Thrown if the owner is null.</exception>
		IStaticContext IHasEditableExpression.StaticContext
		{
			get
			{
				if (Owner != null)
					return Owner.StaticContext;
				else
					throw new InvalidOperationException(HandlesStringRes.sOwnerNotSet);
			}
		}

		ExpressionKind IHasEditableExpression.ExpressionKind
		{
			get {return ExpressionKind.Ocl;}
		}
		#endregion

		[Editor("Borland.Eco.Handles.Design.OclEditorWakeup, Borland.Eco.Handles.Design", typeof(UITypeEditor))]
		[DefaultValue("")]
		[Browsable(true)]
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Visible)]
		[LocalizableCategory(typeof(HandlesStringRes), "sCategoryDerivation")]
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyDerivationExpression")]
		public override string Expression // Doubles for IHasEditableExpression
		{
			get { return m_Expression; }
			set
			{
				if (Expression != value)
				{
					m_Expression = value;
					Changed();
				}
			}
		}

		///<summary>
		///This property is always false for an OclColumn.
		///For a column deriving its value by an event, use an <see cref="EventDerivedColumn"/>.
		///</summary>
		public override bool EventDerivedValue
		{
			get { return false; }
			set {}
		}
	}

	/// <summary>
	/// <para>This class is for backwards comatibility only</para>
	/// </summary>
	[Serializable()]
	[TypeConverter(typeof(ColumnConverter))]
	public class Column: OclColumn
	{
		private bool m_EventDerivedValue;

		public override bool EventDerivedValue
		{
			get { return m_EventDerivedValue; }
			set
			{
				m_EventDerivedValue = value;
				if (value)
					IsReadOnly = true;
			}
		}
	}

	/// <summary>
	/// <para> Class describing an additional property to be added to the items in the rendered list, which gets
	/// its value through ElementHandle.DeriveValue</para>
	/// </summary>

	[Serializable()]
	[TypeConverter(typeof(EventDerivedColumnConverter))]
	public class EventDerivedColumn: AbstractColumn, IHasTypeSystem
	{

		///<summary>
		///The property Expression is not settable in the class EventDerivedColumn.
		///For a column deriving its value using an OCL expression, use an <see cref="OclColumn"/>.
		///</summary>
		public override string Expression
		{
			get { return "n/a"; } // Do not localize
			set { }
		}

		///<summary>
		///The property EventDerivedValue is not settable in the class EventDerivedColumn. It is always true.
		///</summary>
		public override bool EventDerivedValue
		{
			get { return true; }
			set {}
		}

#region IHasTypeSystem
		///<exception cref="InvalidOperationException">Thrown if the owner is null.</exception>
		///<exception cref="InvalidOperationException">Thrown if the owner's StaticContext is null.</exception>
		IEcoTypeSystem IHasTypeSystem.TypeSystem
		{
			get
			{
				if (Owner == null)
					throw new InvalidOperationException(HandlesStringRes.sOwnerNotSet);
				if (Owner.StaticContext == null)
					throw new InvalidOperationException(HandlesStringRes.sRootContextNotSet);
				return Owner.StaticContext.TypeSystem;
			}
		}
		bool IHasTypeSystem.IsHookedUp { get { return Owner != null && Owner.StaticContext != null && Owner.StaticContext.TypeSystem != null; } }
#endregion
		private string m_TypeName = string.Empty;
		/// <summary>
		/// <para>The is the type of the derived value</para>
		/// <para>The eventhandler for ElementHandle.DeriveValue must set e.ResultElement to an element of this type</para>
		/// </summary>
		[LocalizableCategory(typeof(HandlesStringRes), "sCategoryDerivation")]
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyTypeName")]
		[Editor("Borland.Eco.Handles.Design.AnyTypeNameSelectorEditor, Borland.Eco.Handles.Design", typeof(UITypeEditor))]
		[DefaultValue("")]
		public string TypeName
		{
			get { return m_TypeName; }
			set { m_TypeName = value; }
		}
	}
	//ColumnCollection
	[Serializable()]
	[ListBindable(false)]
	[Editor("Borland.Eco.Handles.Design.ColumnCollectionEditor, Borland.Eco.Handles.Design", typeof(UITypeEditor))]
	public sealed class ColumnCollection: System.Collections.CollectionBase
	{
		private IStaticContext m_StaticContext;
		public IStaticContext StaticContext
		{
			get { return m_StaticContext; }
		}
		///<summary>Creates an instance of the class.</summary>
		public ColumnCollection(IStaticContext staticContext): base()
		{
			m_StaticContext = staticContext;
		}
		///<summary>Returns the column at position <paramref name="index"/>.</summary>
		public AbstractColumn this[int index]
		{
			get { return (AbstractColumn)List[index]; }
		}

		///<summary>Returns the column named <paramref name="name"/>.</summary>
		public AbstractColumn this[string name]
		{
			get
			{
				foreach (AbstractColumn c in List)
					if (c.Name == name)
						return c;
				return null;
			}
		}

		///<summary>Adds a <paramref name="column"/> to the collection.</summary>
		public void Add(AbstractColumn column)
		{
			List.Add(column);
		}

		///<summary>Returns the index of <paramref name="value"/> within the collection.</summary>
		public int IndexOf(System.Object value)
		{
			return List.IndexOf(value);
		}
		///<summary>Adds an array of <paramref name="columns"/> to the collection.</summary>
		public void AddRange(AbstractColumn[] columns)
		{
			foreach(AbstractColumn aColumn in columns)
				Add(aColumn);
		}
		///<summary>Adds an array of <paramref name="columns"/> to the collection.</summary>
		///<remark>For backwards compatibility.</remark>
		public void AddRange(Column[] columns)
		{
			foreach(Column aColumn in columns)
				Add(aColumn);
		}

		internal Boolean NameIsUnique(String name)
		{
			for (int i = 0; i < Count; i++)
				if (string.Compare(name, this[i].Name, true, CultureInfo.InvariantCulture) == 0) // case insensitive
					return false;
			return true;
		}

		private string GetUniqueName()
		{
			String value;

			int i = 1;
			do
			{
				value = "New" + i.ToString(CultureInfo.InvariantCulture); // Do not localize
				i++;
			} while (!NameIsUnique(value));
			return value;
		}

		internal void Changed()
		{
			if (OnChange != null)
				OnChange(this, EventArgs.Empty);
		}

		///<exception cref="InvalidOperationException">Thrown if the name of the added column exists in the collection.</exception>
		protected override void OnInsert(int index, object value)
		{
			base.OnInsert(index, value);

			AbstractColumn aColumn = (AbstractColumn) value;

			if (aColumn.Name == null || aColumn.Name.Length == 0)
				aColumn.Name = GetUniqueName();
			if (!NameIsUnique(aColumn.Name))
				throw new InvalidOperationException(HandlesStringRes.sNameNotUnique(aColumn.Name));
			aColumn.Owner = this;
		}

		protected override void OnInsertComplete(int index, object value)
		{
			base.OnInsertComplete(index,value);
			Changed();
		}

		protected override void OnClearComplete()
		{
			base.OnClear();
			Changed();
		}

		protected override void OnRemoveComplete(int index, object value)
		{
			base.OnRemoveComplete(index,value);
			Changed();
		}
		public event EventHandler OnChange;
	}
}
